/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#define FTP_BUILD
#include "mod_ftp.h"

/* ftp_message_generate: Function to transform meta-characters
 *
 * Arguments: fc - The ftp connection
 *            inbuf - The input buffer
 *            outbuf - The output buffer
 *            outlen - The length of the output buffer
 *
 * Returns: nadda
 */
void ftp_message_generate(ftp_connection *fc, const char *inbuf,
                               char *outbuf, size_t outlen)
{
    conn_rec *c = fc->connection;
    const char *inptr = inbuf;
    char *outptr = outbuf;
    char time_str[APR_CTIME_LEN];
    char *remote_host, *local_host;

    while ((outlen > 1) && (*inptr != '\0')) {
        if (*inptr != '%') {
            *outptr++ = *inptr;
            outlen -= 1;
        }
        else {
            switch (*++inptr) {
            case 'T':
                apr_ctime(time_str, apr_time_now());
                apr_cpystrn(outptr, time_str, outlen);
                break;
            case 'C':
                apr_snprintf(outptr, outlen, "%s", fc->cwd);
                break;
            case 'h':
                apr_getnameinfo(&remote_host, c->remote_addr, 0);
                apr_snprintf(outptr, outlen, "%s", remote_host);
                break;
            case 'L':
                apr_getnameinfo(&local_host, c->local_addr, 0);
                apr_snprintf(outptr, outlen, "%s", local_host);
                break;
            case 'E':
                apr_snprintf(outptr, outlen, "%s",
                             fc->orig_server->server_admin);
                break;
            case 'a':
                apr_snprintf(outptr, outlen, "%s", c->local_ip);
                break;
            case 'A':
                apr_snprintf(outptr, outlen, "%s", c->remote_ip);
                break;
            case 'u':
                apr_snprintf(outptr, outlen, "%s", fc->user);
                break;
            case 'f':
                apr_snprintf(outptr, outlen, "%d", fc->files);
                break;
            case 't':
                apr_snprintf(outptr, outlen, "%" APR_OFF_T_FMT, fc->bytes);
                break;
            case 'b':
                apr_snprintf(outptr, outlen, "%" APR_OFF_T_FMT, fc->traffic);
                break;
            case 'x':
                apr_snprintf(outptr, outlen, "%d", fc->transfers);
                break;
            case '%':
                *outptr++ = '%';
                outlen -= 1;
                *outptr = '\0';
                break;
            default:
                *outptr++ = '%';
                outlen -= 1;
                if (outlen > 1) {
                    *outptr++ = *inptr;
                    outlen -= 1;
                }
                *outptr = '\0';
                break;
            }
            outptr[outlen - 1] = '\0';
            while (*outptr) {
                outptr++;
                outlen -= 1;
            }
        }
        inptr++;
    }
    if (outlen > 0) {
        *outptr = '\0';
    }
}
