/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#define FTP_BUILD
#include "mod_ftp.h"
#include "apr_fnmatch.h"
#include "ap_mpm.h"             /* For MPM query interface */

#include <sys/stat.h>           /* For file perms */


/* Warning; the *arg is consumed, manipulated and must have the same lifetime
 * as the desired *addr results!
 */
int ftp_eprt_decode(apr_int32_t *family, char **addr, apr_port_t *port,
                        char *arg)
{
    char *argv, delim = *arg;

    if (delim <= ' ' || delim > 126)
        return FTP_REPLY_SYNTAX_ERROR;

    ++arg;
    argv = arg;
    while (isdigit(*arg))
        ++arg;
    if (*arg != delim)
        return FTP_REPLY_SYNTAX_ERROR;
    *(arg++) = '\0';
    if (*argv) {
        if (strcmp(argv, "1") == 0)
            *family = APR_INET;
#if APR_HAVE_IPV6
        else if (strcmp(argv, "2") == 0)
            *family = APR_INET6;
#endif
        else if (isdigit(argv[0]))
            return FTP_REPLY_BAD_PROTOCOL;
        else
            return FTP_REPLY_SYNTAX_ERROR;
    }

    argv = arg;
    if (*arg == delim)
        return FTP_REPLY_SYNTAX_ERROR;
    if (*family == APR_INET) {
        while (isdigit(*arg) || (*arg == '.'))
            ++arg;
    }
#if APR_HAVE_IPV6
    else if (*family == APR_INET6) {
        while (isxdigit(*arg) || (*arg == ':'))
            ++arg;
        while (isdigit(*arg) || (*arg == '.'))
            ++arg;
    }
#endif
    else
        return FTP_REPLY_BAD_PROTOCOL;
    if (*arg != delim)
        return FTP_REPLY_SYNTAX_ERROR;
    *(arg++) = '\0';
    *addr = argv;

    argv = arg;
    if (*arg == delim)
        return FTP_REPLY_SYNTAX_ERROR;
    while (isdigit(*arg))
        ++arg;
    if (*arg != delim)
        return FTP_REPLY_SYNTAX_ERROR;
    *(arg++) = '\0';
    if (*argv)
        *port = atoi(argv);

    if (*arg)
        return FTP_REPLY_SYNTAX_ERROR;
    return FTP_REPLY_COMMAND_OK;
}

static char *ftp_modestring_get(char *mode, apr_filetype_e typ,
                                     apr_fileperms_t perms)
{

#ifdef WIN32
    perms = ftp_unix_mode2perms(0);
#endif                          /* WIN32 */

    if (perms < 0 || perms >= FTP_MAX_MODESTRING) {
        return FTP_UNKNOWN_MODESTRING;  /* see mod_ftp.h */
    }

    if (typ == APR_DIR) {
        mode[0] = 'd';
    }
    if (perms & APR_UREAD) {
        mode[1] = 'r';
    }
    if (perms & APR_UWRITE) {
        mode[2] = 'w';
    }
    if (perms & APR_UEXECUTE) {
        mode[3] = 'x';
    }
    if (perms & APR_USETID) {
        mode[3] = 's';
    }
    if (perms & APR_GREAD) {
        mode[4] = 'r';
    }
    if (perms & APR_GWRITE) {
        mode[5] = 'w';
    }
    if (perms & APR_GEXECUTE) {
        mode[6] = 'x';
    }
    if (perms & APR_GSETID) {
        mode[6] = 's';
    }
    if (perms & APR_WREAD) {
        mode[7] = 'r';
    }
    if (perms & APR_WWRITE) {
        mode[8] = 'w';
    }
    if (perms & APR_WEXECUTE) {
        mode[9] = 'x';
    }
    if (perms & APR_WSTICKY) {
        mode[9] = 't';
    }
    return mode;
}


/* ftp_direntry_make: Fill out a directory entry structure from the
 *                    directory being requested and a filename.
 *
 * Arguments: r       - The request record for the directory index
 *            name    - The filename to be checked.
 *            pattern - Pattern to get a directory listing for. If NULL
 *                      we do not GLOB.
 *
 * Returns: ftp_direntry structure on success, NULL otherwise.
 */
static struct ftp_direntry *ftp_direntry_make(request_rec *r,
                                                           const char *name,
                                                        const char *pattern)
{
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    struct ftp_direntry *dirent;
    request_rec *rr;
    const char *test, *sl;
    char mode[FTP_MODESTRING_LEN] = "----------";

    for (test = name; (sl = ap_strchr_c(test, '/')); test = sl + 1)
         /* noop */ ;

    if (!strcmp("..", test)) {
        return NULL;
    }

#ifdef FTP_DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Name    - fdm: %s", name);
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Pattern - fdm: %s", pattern);
#endif

    if (pattern && *pattern &&
        (apr_fnmatch(pattern, name, APR_FNM_PATHNAME | APR_FNM_PERIOD) != APR_SUCCESS)) {
#ifdef FTP_DEBUG
        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "no match");
#endif
        return NULL;
    }

    rr = ap_sub_req_lookup_file(name, r, NULL);

    if ((rr->finfo.filetype != 0) &&
        ((rr->status == HTTP_OK) || (rr->status == HTTP_MOVED_PERMANENTLY) ||
         (rr->status == HTTP_UNAUTHORIZED &&
          fsc->options & FTP_OPT_SHOWUNAUTH)) &&
        (rr->uri != NULL)) {
        apr_time_exp_t xt;
        apr_size_t retcode;
        char *lasts;

        /* We don't mess with char returned here, so its fine to cast */
        lasts = ap_strrchr((char *) name, '/');

        dirent = apr_pcalloc(r->pool, sizeof(ftp_direntry));
        dirent->next = NULL;
        dirent->name = apr_pstrdup(r->pool, lasts + 1);
        dirent->nlink = rr->finfo.nlink;
        dirent->size = rr->finfo.size;
        dirent->csize = rr->finfo.csize;
        dirent->modestring = apr_pstrdup(r->pool,
                                         ftp_modestring_get(
                                                            mode,
                                                         rr->finfo.filetype,
                                                       rr->finfo.protection)
            );

        /*
         * If FTPOptions RemoveUserGroup is set, we don't bother looking up
         * user and group information for each file
         */
        if (fsc->options & FTP_OPT_REMOVEUSERGROUP) {
            dirent->username = apr_psprintf(r->pool, "%d", rr->finfo.user);
            dirent->groupname = apr_psprintf(r->pool, "%d", rr->finfo.group);
        }
        else {
            if ((apr_uid_name_get(&dirent->username, rr->finfo.user, r->pool)
                 !=APR_SUCCESS) || (!dirent->username)
                || (!dirent->username[0])) {
                dirent->username = apr_psprintf(r->pool, "%d", rr->finfo.user);
            }
            if ((apr_gid_name_get(&dirent->groupname, rr->finfo.group, r->pool)
                 !=APR_SUCCESS) || (!dirent->groupname)
                || (!dirent->groupname[0])) {
                dirent->groupname = apr_psprintf(r->pool, "%d", rr->finfo.group);
            }
        }

        apr_time_exp_lt(&xt, rr->finfo.mtime);

        if (r->request_time - rr->finfo.mtime >
            180 * 24 * 60 * 60 * APR_USEC_PER_SEC) {
            apr_strftime(dirent->datestring, &retcode,
                         sizeof(dirent->datestring), "%b %e  %Y", &xt);
        }
        else {
            apr_strftime(dirent->datestring, &retcode,
                         sizeof(dirent->datestring), "%b %e %H:%M", &xt);
        }
    }
    else {
        dirent = NULL;
    }

    ap_destroy_sub_req(rr);
    return dirent;
}

/* ftp_dsortf: Used for sorting directory entries. Called by qsort()
 *
 * Arguments: d1 - The first directory entry
 *            d2 - The second directory entry
 *
 * Returns: An integer less than, equal to or greater than zero if
 *          d1 is found, respectively, to be less than, match or
 *          be greater than d2.
 */
static int ftp_dsortf(struct ftp_direntry ** d1,
                          struct ftp_direntry ** d2)
{
    /* Simple sort based on filename */
    return strcmp((*d1)->name, (*d2)->name);
}

/* ftp_direntry_get: Return an array of ftp_direntry structures based
 *                   on the uri stored in the request rec.  An extra
 *                   argument may be passed for pattern matching.
 *
 * Arguments: r       - The request record for the directory index
 *            pattern - Pattern to get a directory listing for.
 *
 * Returns: The sorted array of directory entries on success, NULL otherwise
 */
struct ftp_direntry *ftp_direntry_get(request_rec *r, const char *pattern)
{
    struct ftp_direntry *p, *head, *current, **a;
    apr_dir_t *dir;
    apr_finfo_t finfo;
    apr_status_t rv;
    int num, i;
    char *fname;
    const char *path, *search;

    /*
     * The actual search pattern, used to determine if we should recurse into
     * a directory or not.  If the search pattern is just *, we should not
     * decend.  For search patterns like 'm*', we should.
     */
#ifdef FTP_DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Pattern - start: %s", pattern);
#endif

    path = pattern;
    search = ap_strrchr_c(pattern, '/');

    if (search == NULL) {
        search = ap_strrchr_c(pattern, '\\');
    }
    if (search != NULL) {
        search++;
        path = apr_pstrndup(r->pool, pattern, search - pattern);
    }

#ifdef FTP_DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Path    - end: %s", path);
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Pattern - end: %s", pattern);
#endif

#ifdef WIN32

    /*
     * Win32 will always return sucess on apr_dir_open, which means we have
     * to stat the file to see if the request was made for a directory or
     * file.
     */
    rv = apr_stat(&finfo, path, APR_FINFO_MIN, r->pool);
    if (finfo.filetype != APR_DIR) {
        /* Must be for a single file */
        return ftp_direntry_make(r, path, pattern);
    }
#endif                          /* WIN32 */

    rv = apr_dir_open(&dir, path, r->pool);

    if (rv != APR_SUCCESS) {
        if (APR_STATUS_IS_ENOTDIR(rv)) {
            /* Must be for a single file */
            return ftp_direntry_make(r, path, pattern);
        }
        else {
            return NULL;
        }
    }

    num = 0;
    head = current = NULL;
    while ((rv = apr_dir_read(&finfo, APR_FINFO_DIRENT, dir))
           == APR_SUCCESS) {

        fname = ap_make_full_path(r->pool, path, finfo.name);
        p = ftp_direntry_make(r, fname, pattern);
        if (!p) {
            continue;
        }

        /* Add this entry to the linked list */
        if (head == NULL) {
            head = p;
            p->next = NULL;
            current = p;
        }
        else {
            current->next = p;
            current = p;
        }
        /*
         * We are only going to support single recursive listings this means
         * that requests such as 'ls m*' will print out all files that match,
         * and recurse a single level into directories.
         */

        if (search && (search[0] != '*') && (p->modestring[0] == 'd')) {
            const char *newpattern = apr_pstrcat(r->pool, fname,
                                                 "/*", NULL);
            p->child = ftp_direntry_get(r, newpattern);
        }
        else {
            p->child = NULL;
        }
        num++;
    }

    apr_dir_close(dir);

    /* Sort this mess */
    if (num > 0) {
        a = (struct ftp_direntry **) apr_pcalloc(r->pool,
                                                 num *
                                                 sizeof(ftp_direntry));
        p = head;
        i = 0;
        while (p) {
            a[i++] = p;
            p = p->next;
        }
        num = i;
        qsort((void *) a, num, sizeof(struct ftp_direntry *),
              (int (*) (const void *, const void *)) ftp_dsortf);

        /* Re-construct the list from the sorted list */
        head = a[0];
        current = head;
        for (i = 1; i < num; i++) {
            current->next = a[i];
            current = current->next;
        }
        current->next = NULL;
    }

    return head;
}


/* ftp_set_authorization: set the r->headers_in Authorization header
 *
 * Arguments: r - The request
 *
 * Returns: nada
 */

void ftp_set_authorization(request_rec *r)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    if (fc->user == ftp_unknown_username)
        return;
    r->hostname = apr_pstrdup(r->pool, fc->host);
    r->user = apr_pstrdup(r->pool, fc->user);
    apr_table_setn(r->headers_in, "Host", r->hostname);
    apr_table_setn(r->headers_in, "Authorization", fc->authorization);
}

/* ftp_set_uri: Setup r->uri based on a file argument and user's
 *              current working directory.  We also run the translate
 *              name phase here to set r->filename.
 *
 * Arguments: r - The request
 *            arg - The file argument
 *
 * Returns: nothing
 */
int ftp_set_uri(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    apr_status_t res;

    if (arg[0] == '/') {
        ap_parse_uri(r, arg);
    }
    else {
        ap_parse_uri(r, ap_make_full_path(r->pool, fc->cwd, arg));
    }
    ap_getparents(r->uri);

    /*
     * If the path ended in /., ap_getparents converts it to /, but we really
     * don't want the trailing / there, so remove it.
     */
    if (r->uri[strlen(r->uri) - 1] == '/') {
        r->uri[strlen(r->uri) - 1] = '\0';
    }

    /*
     * Parsed uri is empty if we try a path outside the document
     * root.  For now, just set the uri to /, but I'm sure there
     * is a better way around this.
     *
     * - rpm
     */
    if (r->uri[0] == '\0') {
        r->uri = apr_pstrdup(r->pool, "/");
    }
    res = ap_run_translate_name(r);
    if (res) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                          r->parsed_uri.path);
        return FTP_REPLY_LOCAL_ERROR;
    }
    r->uri = ap_escape_uri(r->pool, r->uri);

    return OK;
}

/* ftp_unix_perms2mode: Translate apr_fileperms_t to mode_t.
 *
 * Arguments: perms - apr_fileperms_t
 *
 * Returns: mode_t
 */
mode_t ftp_unix_perms2mode(apr_fileperms_t perms)
{
    mode_t mode = 0;

#ifndef WIN32
    if (perms & APR_UREAD) {
        mode |= S_IRUSR;
    }
    if (perms & APR_UWRITE) {
        mode |= S_IWUSR;
    }
    if (perms & APR_UEXECUTE) {
        mode |= S_IXUSR;
    }
    if (perms & APR_GREAD) {
        mode |= S_IRGRP;
    }
    if (perms & APR_GWRITE) {
        mode |= S_IWGRP;
    }
    if (perms & APR_GEXECUTE) {
        mode |= S_IXGRP;
    }
    if (perms & APR_WREAD) {
        mode |= S_IROTH;
    }
    if (perms & APR_WWRITE) {
        mode |= S_IWOTH;
    }
    if (perms & APR_WEXECUTE) {
        mode |= S_IXOTH;
    }
#endif                          /* WIN32 */
    return mode;
}

/* ftp_unix_mode2perms: Translate mode_t to apr_fileperms_t.
 *
 * Arguments: mode - mode_t
 *
 * Returns: apr_fileperms_t
 */
apr_fileperms_t ftp_unix_mode2perms(mode_t mode)
{
    apr_fileperms_t perms = 0;
#ifdef WIN32
    perms |= APR_UREAD;
    perms |= APR_UWRITE;
    perms |= APR_UEXECUTE;
    perms |= APR_GREAD;
    perms |= APR_GEXECUTE;
    perms |= APR_WREAD;
    perms |= APR_WEXECUTE;

#else
    if (mode & S_IRUSR) {
        perms |= APR_UREAD;
    }
    if (mode & S_IWUSR) {
        perms |= APR_UWRITE;
    }
    if (mode & S_IXUSR) {
        perms |= APR_UEXECUTE;
    }
    if (mode & S_IRGRP) {
        perms |= APR_GREAD;
    }
    if (mode & S_IWGRP) {
        perms |= APR_GWRITE;
    }
    if (mode & S_IXGRP) {
        perms |= APR_GEXECUTE;
    }
    if (mode & S_IROTH) {
        perms |= APR_WREAD;
    }
    if (mode & S_IWOTH) {
        perms |= APR_WWRITE;
    }
    if (mode & S_IXOTH) {
        perms |= APR_WEXECUTE;
    }
#endif                          /* WIN32 */
    return perms;
}

/* ftp_toupper: Convert a string to uppercase
 *
 * Arguments: s - The string
 *
 * Returns: The capitialized string.
 */
char *ftp_toupper(apr_pool_t *p, const char *s)
{
    char *upper = apr_pstrdup(p, s);
    char *pos = upper;

    while (*pos != '\0') {
        *pos = apr_toupper(*pos);
        pos++;
    }

    return upper;
}

/* ftp_check_maxclients: Check the scoreboard for other available servers.
 *
 * Arguments: r - The current request
 *
 * Returns: 0 if we find a server, 1 otherwise.
 */
int ftp_check_maxclients(request_rec *r)
{
    int hard_server_limit, hard_thread_limit;
    int i, j;
    worker_score *scoreboard;

    ap_mpm_query(AP_MPMQ_HARD_LIMIT_DAEMONS, &hard_server_limit);
    ap_mpm_query(AP_MPMQ_HARD_LIMIT_THREADS, &hard_thread_limit);

    for (i = 0; i < hard_server_limit; i++) {
        for (j = 0; j < hard_thread_limit; j++) {
#if ((AP_SERVER_MAJORVERSION_NUMBER < 3) && (AP_SERVER_MINORVERSION_NUMBER < 3))
            scoreboard = ap_get_scoreboard_worker(i, j);
#else
            scoreboard = ap_get_scoreboard_worker_from_indexes(i, j);
#endif
            if (scoreboard->status == SERVER_READY)
                return 0;
        }
    }

    /*
     * We are the only available server, so go ahead.  Maybe this should be
     * optimized out so it's only in debug builds?
     */
    if (ap_exists_config_define("ONE_PROCESS")) {
        return 0;
    }

    ap_log_rerror(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, 0, r,
                  "Maximum number of FTP sessions reached.");
    return 1;
}
