/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#define FTP_BUILD
#include "mod_ftp.h"

/* Reimplement the core i/o filters to insert two-channel
 * data socket buckets for send or retrieve, monitoring
 * the control channel for sideband ABOR-like commands.
 *
 * These filters sit below
 */

static apr_status_t datasock_bucket_read(apr_bucket * a, const char **str,
                                    apr_size_t * len, apr_read_type_e block)
{
    ftp_connection *fc = a->data;
    apr_socket_t *sd = fc->datasock;
    char *buf;
    apr_status_t rv;
    apr_interval_time_t timeout;
    apr_interval_time_t polltimeout;
    apr_pollfd_t pollset[2];
    int n;

    /* command socket heard already?  (e.g. already polled) */
    rv = ftp_read_ahead_request(fc);

    if ((rv != APR_SUCCESS) &&
        !(APR_STATUS_IS_EINTR(rv) || (APR_STATUS_IS_EAGAIN(rv)) ||
          APR_STATUS_IS_EOF(rv))) {
        return rv;
    }

    apr_socket_timeout_get(sd, &timeout);

    if (block == APR_NONBLOCK_READ) {
        apr_socket_timeout_set(sd, 0);
        polltimeout = 0;
    }
    else {
        polltimeout = timeout;
    }

    *str = NULL;
    /* buf allocation must mirror *len initalization below */
    buf = apr_bucket_alloc(APR_BUCKET_BUFF_SIZE, a->list);

    pollset[0].desc_type = APR_POLL_SOCKET;
    pollset[0].desc.s = fc->datasock;
    pollset[0].reqevents = APR_POLLIN;  /* APR_POLLOUT for write */
    pollset[1].desc_type = APR_POLL_SOCKET;
    pollset[1].desc.s = fc->cntlsock;
    pollset[1].reqevents = (APR_POLLIN | APR_POLLPRI);
    pollset[1].rtnevents = 0;   /* For poll(1), pretend poll answered this */

    /* XXX: evil, no apr_socket_pool_get accessor available, though */
    pollset[1].p = pollset[0].p = *((apr_pool_t **) fc->datasock);

    do {
        /* Unset so we don't trip over the len when we don't call recv */
        *len = 0;

        /*
         * If we have already assembled our next_request, don't bother
         * polling the control connection - we won't read ahead two commands
         */
        rv = apr_poll(pollset, fc->next_request ? 1 : 2,
                      &n, polltimeout);
        /* pedantic sanity check, this should never happen */
        if ((rv == APR_SUCCESS) && (n < 0)) {
            rv = APR_EGENERAL;
        }

        if (rv != APR_SUCCESS) {
            /*
             * The loss of either socket here means the death of the data
             * connection. Unset the length to avoid looping
             */
            break;
        }
        if (pollset[1].rtnevents & (APR_POLLIN | APR_POLLPRI)) {
            /* command socket heard - but with a full line yet? */
            rv = ftp_read_ahead_request(fc);

            if ((rv != APR_SUCCESS) &&
                !(APR_STATUS_IS_EINTR(rv) || (APR_STATUS_IS_EAGAIN(rv)) ||
                  APR_STATUS_IS_EOF(rv))) {
                apr_bucket_free(buf);
                return rv;
            }
        }

        if (pollset[0].rtnevents & APR_POLLIN) {
            /* *len must mirror the apr_bucket_alloc above. */
            *len = APR_BUCKET_BUFF_SIZE;
            rv = apr_socket_recv(sd, buf, len);
        }
    } while (APR_STATUS_IS_EINTR(rv)
             || (APR_STATUS_IS_EAGAIN(rv) && (block == APR_BLOCK_READ)));
    /*
     * EINTR, above, is obvious, EAGAIN is less so - win32 (perhaps others)
     * can trigger POLLIN a little too early, before the recieved packet has
     * actually been disassembled - so loop again.
     */

    if (block == APR_NONBLOCK_READ) {
        apr_socket_timeout_set(sd, timeout);
    }

    if (rv != APR_SUCCESS && !APR_STATUS_IS_EOF(rv)) {
        apr_bucket_free(buf);
        return rv;
    }
    /*
     * If there's more to read we have to keep the rest of the socket
     * for later. XXX: Note that more complicated bucket types that
     * refer to data not in memory and must therefore have a read()
     * function similar to this one should be wary of copying this
     * code because if they have a destroy function they probably
     * want to migrate the bucket's subordinate structure from the
     * old bucket to a raw new one and adjust it as appropriate,
     * rather than destroying the old one and creating a completely
     * new bucket.
     *
     * Even if there is nothing more to read, don't close the socket here
     * as we have to use it to send any response :)  We could shut it
     * down for reading, but there is no benefit to doing so.
     */
    if (*len > 0) {
        apr_bucket_heap *h;
        /* Change the current bucket to refer to what we read */
        a = apr_bucket_heap_make(a, buf, *len, apr_bucket_free);
        h = a->data;
        h->alloc_len = APR_BUCKET_BUFF_SIZE;    /* note the real buffer size */
        *str = buf;
        APR_BUCKET_INSERT_AFTER(a, ftp_bucket_datasock_create(fc, a->list));
    }
    else {
        apr_bucket_free(buf);
        a = apr_bucket_immortal_make(a, "", 0);
        *str = a->data;
    }
    return APR_SUCCESS;
}

static apr_bucket *ftp_bucket_datasock_make(apr_bucket * b, ftp_connection *fc)
{
    /*
     * Note that typically the socket is allocated from the connection pool
     * so it will disappear when the connection is finished.
     */
    b->type = &ftp_bucket_type_datasock;
    b->length = (apr_size_t) (-1);
    b->start = -1;
    b->data = fc;

    return b;
}

apr_bucket *ftp_bucket_datasock_create(ftp_connection *fc,
                                                  apr_bucket_alloc_t * list)
{
    apr_bucket *b = apr_bucket_alloc(sizeof(*b), list);

    APR_BUCKET_INIT(b);
    b->free = apr_bucket_free;
    b->list = list;
    return ftp_bucket_datasock_make(b, fc);
}

const apr_bucket_type_t ftp_bucket_type_datasock = {
    "DATASOCK",
    5,
    APR_BUCKET_DATA,
    apr_bucket_destroy_noop,
    datasock_bucket_read,
    apr_bucket_setaside_notimpl,
    apr_bucket_split_notimpl,
    apr_bucket_copy_notimpl
};


apr_status_t ftp_data_out_filter(ap_filter_t *f, apr_bucket_brigade *bb)
{
    ftp_connection *fc = f->ctx;
    ftp_server_config *fsc;
    conn_rec *c = fc->connection;
    apr_status_t rv;
    apr_interval_time_t polltimeout;
    apr_pollfd_t pollset[2];
    apr_bucket_brigade *next_bb;
    apr_bucket *e;
    int n;

    /* command socket heard already?  (e.g. already polled) */
    rv = ftp_read_ahead_request(fc);

    if ((rv != APR_SUCCESS) &&
        !(APR_STATUS_IS_EINTR(rv) || (APR_STATUS_IS_EAGAIN(rv)) ||
          APR_STATUS_IS_EOF(rv))) {
        f->c->aborted = 1;
        return rv;
    }

    fsc = ftp_get_module_config(fc->orig_server->module_config);

    while (!APR_BRIGADE_EMPTY(bb)) {

        /*
         * The brigade can be far to 'chunky' to respect the ABOR tests we
         * perform below.  Dice up the raw response stream into chuncks
         * (48000 bytes by default) and and check between each chunk for a
         * control channel command.  At 48000 this is at least 1/minute on a
         * very slow 9600 baud line.
         */
        rv = apr_brigade_partition(bb, fsc->data_block_size, &e);
        if (rv != APR_SUCCESS && rv != APR_INCOMPLETE) {
            return rv;
        }
        next_bb = apr_brigade_split(bb, e);

        /*
         * Poll to see if we are are prepared to pass on the brigade and the
         * client hasn't ABORted us yet.
         */
        apr_socket_timeout_get(fc->datasock, &polltimeout);

        pollset[0].desc_type = APR_POLL_SOCKET;
        pollset[0].desc.s = fc->datasock;
        pollset[0].reqevents = APR_POLLOUT;
        pollset[1].desc_type = APR_POLL_SOCKET;
        pollset[1].desc.s = fc->cntlsock;
        pollset[1].reqevents = (APR_POLLIN | APR_POLLPRI);
        pollset[1].rtnevents = 0;       /* For poll(1), pretend poll answered
                                         * this */

        /* XXX: evil, no apr_socket_pool_get accessor available, though */
        pollset[1].p = pollset[0].p = *((apr_pool_t **) fc->datasock);

        do {
            /*
             * If we have already assembled our next_request, don't bother
             * polling the control connection - we won't read ahead two
             * commands
             */
            rv = apr_poll(pollset, fc->next_request ? 1 : 2,
                          &n, polltimeout);
            /* pedantic sanity check, this should never happen */
            if ((rv == APR_SUCCESS) && (n < 0)) {
                rv = APR_EGENERAL;
            }

            if (rv != APR_SUCCESS) {
                /*
                 * The loss of either socket here means the death of the data
                 * connection.
                 */
                break;
            }

            if (pollset[1].rtnevents & (APR_POLLIN | APR_POLLPRI)) {
                /* command socket heard - but with a full line yet? */
                rv = ftp_read_ahead_request(fc);

                if ((rv != APR_SUCCESS) &&
                  !(APR_STATUS_IS_EINTR(rv) || (APR_STATUS_IS_EAGAIN(rv)) ||
                    APR_STATUS_IS_EOF(rv))) {
                    f->c->aborted = 1;
                    return rv;
                }
            }

            if ((rv == APR_SUCCESS) && (pollset[0].rtnevents & APR_POLLOUT)) {
                break;
            }
        } while (APR_STATUS_IS_EINTR(rv) || APR_STATUS_IS_EAGAIN(rv));
        /*
         * EINTR, above, is obvious, EAGAIN is less so - win32 (perhaps
         * others) can trigger POLLIN a little too early, before the recieved
         * packet has actually been disassembled - so loop again.
         */

        if (c->aborted || f->c->aborted) {
            return AP_FILTER_ERROR;
        }

        rv = ap_pass_brigade(f->next, bb);
        if (rv != APR_SUCCESS) {
            f->c->aborted = 1;
            return rv;
        }
        bb = next_bb;
    }

    return APR_SUCCESS;
}
