/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#ifndef MOD_FTP_H
#define MOD_FTP_H

#ifdef FTP_BUILD
#ifdef FTP_APXS_BUILD
/* Not defined on Win32/Netware since #undef's aren't really needed */
#include "ftp_config.h"
#endif
#ifndef CORE_PRIVATE
#define CORE_PRIVATE
#endif
#endif /* FTP_BUILD */

#define MODFTP_VERSION "0.9.2"

/* FTP header has a couple of defines... on win32
   rather than including it, we'll just hack em in
   ourselves
 */
#ifdef HAVE_ARPA_FTP_H
#include <arpa/ftp.h>
#else
#define	TYPE_A		1	/* ASCII */
#define	TYPE_I		3	/* image */
#endif

/* Apache headers */
#include "apr_buckets.h"
#include "util_filter.h"
#include "util_ebcdic.h"
#include "apr_strings.h"
#include "apr_hooks.h"
#include "apr_hash.h"
#include "apr_user.h"
#include "apr_lib.h"
#include "apr_inherit.h"
#include "apr_poll.h"
#include "ap_config.h"
#include "ap_mmn.h"
#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_main.h"
#include "http_core.h"
#include "http_connection.h"
#include "http_request.h"
#include "http_protocol.h"
#include "scoreboard.h"

#ifdef FTP_BUILD
#ifdef FTP_APXS_BUILD
#include "mod_log_config.h"
#else
#include "../../modules/loggers/mod_log_config.h"
#endif
#endif

#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
/* FTP Reply Codes */

#define FTP_REPLY_RESTART_MARKER          110
#define FTP_REPLY_SERVICE_READY_IN_N_MIN  120
#define FTP_REPLY_DATA_ALREADY_OPEN       125
#define FTP_REPLY_FILE_STATUS_OK          150
#define FTP_REPLY_COMMAND_OK              200
#define FTP_REPLY_NOT_IMPLEMENTED         202
#define FTP_REPLY_SYSTEM_STATUS           211
#define FTP_REPLY_DIRECTORY_STATUS        212
#define FTP_REPLY_FILE_STATUS             213
#define FTP_REPLY_HELP_MESSAGE            214
#define FTP_REPLY_SYSTEM_TYPE             215
#define FTP_REPLY_SERVICE_READY           220
#define FTP_REPLY_CONTROL_CLOSE           221
#define FTP_REPLY_DATA_OPEN               225
#define FTP_REPLY_DATA_CLOSE              226
#define FTP_REPLY_PASSIVE_MODE            227
#define FTP_REPLY_EXTENDED_PASSIVE_MODE   229
#define FTP_REPLY_USER_LOGGED_IN          230
#define FTP_REPLY_USER_LOGGED_IN_SECURE   232
#define FTP_REPLY_SECURITY_EXCHANGE_DONE  234
#define FTP_REPLY_COMPLETED               250
#define FTP_REPLY_PATH_CREATED            257
#define FTP_REPLY_USER_OK                 331
#define FTP_REPLY_NEED_ACCOUNT            332
#define FTP_REPLY_PENDING                 350
#define FTP_REPLY_SERVICE_NOT_AVAILABLE   421
#define FTP_REPLY_CANNOT_OPEN_DATACONN    425
#define FTP_REPLY_TRANSFER_ABORTED        426
#define FTP_REPLY_FILE_BUSY               450
#define FTP_REPLY_LOCAL_ERROR             451
#define FTP_REPLY_INSUFFICIENT_SPACE      452
#define FTP_REPLY_COMMAND_UNRECOGNIZED    500
#define FTP_REPLY_SYNTAX_ERROR            501
#define FTP_REPLY_COMMAND_NOT_IMPLEMENTED 502
#define FTP_REPLY_BAD_SEQUENCE            503
#define FTP_REPLY_COMMAND_NOT_IMPL_PARAM  504
#define FTP_REPLY_BAD_PROTOCOL            522
#define FTP_REPLY_NOT_LOGGED_IN           530
#define FTP_REPLY_NEED_ACCOUNT_TO_STORE   532
#define FTP_REPLY_PROT_NOT_SUPPORTED      536
#define FTP_REPLY_FILE_NOT_FOUND          550
#define FTP_REPLY_PAGE_TYPE_UNKNOWN       551
#define FTP_REPLY_EXEEDED_STORAGE         552
#define FTP_REPLY_FILENAME_NOT_ALLOWED    553

/*
 * FTP_NEED_* - Values used for filter bitmasks
 *
 */

#define FTP_NEED_BYTERANGE      (1 << 1)
#define FTP_NEED_CONTENT_LENGTH (1 << 2)
#define FTP_NEED_CRLF           (1 << 4)
#define FTP_NEED_DATA_OUT       (1 << 5)
/*
 * FTP_MSG_* - Text for sending back to the client.  These include 
 *             metacharacters to enforce rigidity of layout.
 */

#define FTP_MSG_PERM_DENIED  "%s: Permission denied"
#define FTP_MSG_NOT_A_DIR    "%s: Not a directory"
#define FTP_MSG_NOSUCHFILE   "%s: No such file or directory"
#define FTP_MSG_OPENASCII    "Opening ASCII mode data connection for file list"
#define FTP_MSG_SUCCESS      "%s: Command successful"
#define FTP_MSG_NOTPLAIN     "%s: Not a plain file"
#define FTP_MSG_DIR_CREAT    "\"%s\" new directory created"
#define FTP_MSG_DIR_CUR      "\"%s\" is current directory"
#define FTP_MSG_HELP_SYNTAX  "Syntax: %s %s"
#define FTP_MSG_HELP         "Direct comments to %s"
#define FTP_MSG_NOTIMPL      "%s: Command not implemented"
#define FTP_MSG_NOTALLOWED   "Permission denied: %s not allowed here"

#define FTP_DEFAULT_DBFILE   "logs/ftplogins"
/*
 * FTP_OPT_* - FTP server options
 *
 */
#define FTP_OPT_NONE             0
#define FTP_OPT_REQUIRESSL      (1 << 0) /* Require AUTH before USER/PASS */
#define FTP_OPT_CHECKMAXCLIENTS (1 << 1) /* Check if the server is full */
#define FTP_OPT_REMOVEUSERGROUP (1 << 2) /* Fill in user/group information
                                          * on directory listings */
#define FTP_OPT_NLSTSHOWDIRS    (1 << 3) /* Show directories on NLST 
                                          * requests */
#define FTP_OPT_NLSTISLIST      (1 << 4) /* Maps NLST directly to LIST */
#define FTP_OPT_CREATEHOMEDIRS  (1 << 5) /* Create home dirs on the fly */
#define FTP_OPT_SHOWUNAUTH      (1 << 6) /* Show unauthorized files in LIST */
#define FTP_OPT_LISTISNLST      (1 << 7) /* Maps LiST directly to NLST */
#define FTP_OPT_ALLOWPROXYPORT  (1 << 8) /* Data connect to other than client */
#define FTP_OPT_ALLOWPROXYPASV  (1 << 9) /* Data listens to other than client */
#define FTP_OPT_VHOST_BY_USER  (1 << 10) /* Parse USER for @hostname vhost */
#define FTP_OPT_STRIP_HOSTNAME (1 << 11) /* Strip @hostname from username */

/* Supported FTP AUTH mechanisms. */
#define FTP_AUTH_NONE         0
#define FTP_AUTH_SSL          1
#define FTP_AUTH_TLS          2

/* Data protection levels */
#define FTP_PROT_CLEAR        0
#define FTP_PROT_SAFE         1
#define FTP_PROT_CONFIDENTIAL 2
#define FTP_PROT_PRIVATE      3

/* SSL Filter name */
#define FTP_SSL_FILTER        "SSL/TLS Filter"

#if defined(WIN32) && defined(FTP_DECLARE_EXPORT)
#define FTP_DECLARE_DATA __declspec(dllexport)
#define FTP_DECLARE(fn)  __declspec(dllexport) fn
#elif defined(WIN32)
#define FTP_DECLARE_DATA __declspec(dllimport)
#define FTP_DECLARE(fn)  __declspec(dllimport) fn
#else
#define FTP_DECLARE_DATA 
#define FTP_DECLARE(fn)  fn
#endif

extern module FTP_DECLARE_DATA ftp_module;

typedef struct ftp_server_config ftp_server_config;
typedef struct ftp_dir_config ftp_dir_config;
typedef struct ftp_connection ftp_connection;
typedef struct ftp_direntry ftp_direntry;

typedef enum {
    FTP_LIMIT_OK,
    FTP_LIMIT_HIT_PERUSER,
    FTP_LIMIT_HIT_PERIP,
    FTP_LIMIT_HIT_PERSERVER,
    FTP_LIMIT_ERROR
}
ftp_loginlimit_t;

/* Directory entry structure.  Used for directory listings */
struct ftp_direntry
{
    const char *name;           /* Just the name of the file (foo.c) */
    const char *modestring;
    char datestring[13];        /* Date srings have a fixed length */
    char *username;
    char *groupname;
    apr_off_t size; 
    apr_off_t csize;
    apr_int32_t  nlink;
    struct ftp_direntry *child; /* For descending */
    struct ftp_direntry *next;
};

/* The main ftp server configuration */
struct ftp_server_config
{
    int enabled;
    int options;
    int implicit_ssl;
    int timeout_login;
    int timeout_idle;
    int timeout_data;
    int active_min;
    int active_max;
    int pasv_min;
    int pasv_max;
    char *pasv_addr;
    char *pasv_bindaddr;
    int pasv_bindfamily;
    int epsv_ignore_family;
    int data_block_size;
    const char *banner_message;
    int banner_message_isfile;
    const char *exit_message;
    int exit_message_isfile;
    const char *homedir;
    const char *docrootenv;
    int jailuser;
    int max_login_attempts;
    int limit_peruser;
    int limit_perip;
    int limit_perserver;
    const char *limitdbfile;
};

struct ftp_dir_config
{
    const char *readme;
    const char *path;
    int readme_isfile;
    apr_fileperms_t fileperms;
    apr_fileperms_t dirperms;
};

/* 
 * FTP connection structure.  Each connection to the server will
 * have an ftp_connection structure associated with it.
 */
struct ftp_connection
{
    conn_rec *connection;     /* The connection */
    server_rec *orig_server;  /* Stow the original server for reset */
    int close_connection;     /* Final control command was processed */
    const char *response_notes; /* Notes for response handling. */

    /* User information */
    apr_pool_t *login_pool;   /* Child of c->pool reset on every USER cmd */
    int logged_in;
    const char *host;
    const char *user;
    const char *authorization;
    int login_attempts;

    /* Security information */
    int auth;                 /* The AUTH type.  One of NONE/SSL/TLS */
    int pbsz;                 /* Data protection buffer size.  Usually zero. */
    int prot;                 /* Data channel protection level. C/S/E/P */
    int is_secure;
    void *ssl_input_ctx;
    void *ssl_output_ctx;

    /* Connection stats */
    int files;                /* STOR/RETR transfer count */
    apr_off_t bytes;          /* Only STOR/RETR style traffic */
    int transfers;            /* All data transfers including LIST/NLST */
    apr_off_t traffic;        /* Total data channel traffic, plus HELP */

    /* Command state */
    char *cwd;                /* APR_PATH_MAX+1 buffer for Current Directory */
    char *rename_from;        /* APR_PATH_MAX+1 buffer for the RNFR command */
    int type;                 /* FTP transmission (TYPE_A/TYPE_I) */
    apr_off_t restart_point;  /* For data transfers */
    int filter_mask;          /* Filters required (CRLF, BYTERANGE, etc) */

    /* Data channel state */
    apr_pool_t *data_pool;    /* Child of c->pool reset on every data cmd */
    int all_epsv;             /* EPSV ALL command issued (refuse all others) */
    apr_socket_t *csock;      /* The local data socket we connect/accept on */
    apr_socket_t *datasock;   /* The remote data socket we send/recv on */
    apr_sockaddr_t *clientsa; /* The remote sa of this data connection */
    apr_time_t passive_created; /* Passive creation time or (-1) for active */

    /* Look-ahead for "next" command */
    apr_socket_t *cntlsock;   /* The control socket to watch for next cmd */
    apr_pool_t *next_pool;    /* Storage to get us to the next request_rec */
    apr_bucket_brigade *next_bb; /* Holding brigade for request look-ahead */
    char *next_request;       /* The next control command */
    apr_size_t next_reqsize;  /* Size of the next control command */
};

#define FTP_DEFAULT_UMASK (APR_GWRITE | APR_WWRITE)
#define FTP_UNSPEC        -2
#define FTP_TIMEOUT_LOGIN 120
#define FTP_TIMEOUT_IDLE  600
#define FTP_TIMEOUT_DATA  300
#define FTP_MAX_LOGINS    3
#define FTP_MAX_TRIES     10 /* Maximum number of times to retry on
                                failed bind() and connect() */

#define FTP_DATA_BLOCK_SIZE  48000 /* Size in bytes to read at a time
                                      on the data channel */

/* Wrappers for module_config
 *
 * mod_ftp.c
 */
#define ftp_get_module_config(v) \
    ap_get_module_config(v, &ftp_module)
#define ftp_set_module_config(v,val) \
    ap_set_module_config(v, &ftp_module, val)

extern ap_filter_rec_t *ftp_input_filter_handle;

/* Routines for handling incoming connections
 * 
 * ftp_connection.c
 */
int ftp_process_connection(conn_rec *c);

/* Placeholder of "unknown" username (not valid for ftp_cmd_pass)
 */
extern const char ftp_unknown_username[];

/* Routines for handling FTP requests 
 *
 * ftp_request.c 
 */
void ftp_process_request(request_rec *r);

/* Routines specific to the FTP Protocol 
 * 
 * ftp_protocol.c 
 */
request_rec *ftp_read_request(ftp_connection *fc);
apr_status_t ftp_read_ahead_request(ftp_connection *fc);
apr_status_t ftp_reply(ftp_connection *fc, ap_filter_t *out_filter, 
                       apr_pool_t *p, int n, int l, const char *fmt, ...);
apr_status_t ftp_show_file(ap_filter_t *out_filter, apr_pool_t *p, int code, 
                           ftp_connection *fc, const char *file);
void ftp_send_response(request_rec *r, int res);
apr_status_t ftp_ssl_init(conn_rec *c);
int ftp_have_ssl(void);

/* Routines specific to the FTP data connections
 * 
 * ftp_data_connection.c 
 */
void ftp_reset_dataconn(ftp_connection *fc);
conn_rec *ftp_open_dataconn(request_rec *r, int write_not_read);

/* FTP command handlers registration.
 */
void ftp_register_core_cmds(apr_pool_t *pool);

/* Routines for handling implementation of login limits.
 * Note that the actual Mutexing functions and DB file
 * functions are non-public.
 *
 * ftp_limitlogin.c
 */
apr_status_t ftp_mutexdb_init(server_rec *s, apr_pool_t *p);
apr_status_t ftp_mutexdb_child_init(server_rec *s, apr_pool_t *p);
apr_status_t ftp_mutexdb_cleanup(void *dummy);
ftp_loginlimit_t ftp_limitlogin_check(const char *user, request_rec *r);
int ftp_limitlogin_loggedout(conn_rec *c);

/* FTP utilities
 *
 * ftp_util.c
 */
int ftp_eprt_decode(apr_int32_t *family, char **addr, apr_port_t *port,
                    char *arg);
struct ftp_direntry *ftp_direntry_get(request_rec *r, const char *pattern);

void ftp_set_authorization(request_rec *r);
int ftp_set_uri(request_rec *r, const char *arg);

#ifdef WIN32
#define mode_t int
#endif

mode_t ftp_unix_perms2mode(apr_fileperms_t perms);
apr_fileperms_t ftp_unix_mode2perms(mode_t mode);

char *ftp_toupper(apr_pool_t *p, const char *s);
int ftp_check_maxclients(request_rec *r);

/* FTP filters
 *
 * ftp_filters.c
 */
apr_status_t ftp_data_out_filter(ap_filter_t * f, apr_bucket_brigade * bb);
apr_status_t ftp_crlf_filter(ap_filter_t *f, apr_bucket_brigade *bb);
apr_status_t ftp_protocol_filter(ap_filter_t *f, apr_bucket_brigade *b,
                              ap_input_mode_t mode, apr_read_type_e block,
                              apr_off_t readbytes);
apr_bucket *ftp_bucket_datasock_create(ftp_connection *fc,
                                       apr_bucket_alloc_t *list);
extern const apr_bucket_type_t ftp_bucket_type_datasock;

/* FTP logging hooks into mod_log_config
 *
 * ftp_log.c
 */
const char *ftp_log_transfer_mode(request_rec *r, char *a);
const char *ftp_log_action_flags(request_rec *r, char *a);
const char *ftp_log_transfer_direction(request_rec *r, char *a);
const char *ftp_log_accessed_anonymously(request_rec *r, char *a);
const char *ftp_log_service_name(request_rec *r, char *a);
const char *ftp_log_auth_method(request_rec *r, char *a);
const char *ftp_log_auth_user_id(request_rec *r, char *a);

/* dirlist stuff */
#define FTP_MODESTRING_LEN 11  /* # of chars in a modestring, with \0 */
#define FTP_UNKNOWN_MODESTRING "??????????"
#define FTP_MAX_MODESTRING ((APR_UREAD  | APR_UWRITE | APR_UEXECUTE | \
                             APR_GREAD  | APR_GWRITE | APR_GEXECUTE | \
                             APR_WREAD  | APR_WWRITE | APR_WEXECUTE | \
                             APR_USETID | APR_GSETID | APR_WSTICKY) + 1)

/* FTP message functions for dealing with client messages
 *
 * ftp_message.c
 */
void ftp_message_generate(ftp_connection *fc, const char *inbuf,
                          char *outbuf, size_t outlen);

/* inet_pton functionallity ripped straight out of APR
 *
 * ftp_inet_pton.c
 */
int ftp_inet_pton(int af, const char *src, void *dst);

/* FTP hooks */

/* Flags for each FTP command handler.
 *
 * FTP_TAKE0 - This command takes no arguments.
 * FTP_TAKE1 - This command takes a single argument.
 * FTP_NEED_LOGIN - The user needs to be logged in to execute this command.
 * FTP_DATA_INTR - The current data transfer is interrupted by this command.
 */
#define FTP_TAKE0            (1 << 0)
#define FTP_TAKE1            (1 << 1)
#define FTP_NEED_LOGIN       (1 << 2)
#define FTP_DATA_INTR        (1 << 3)

/* FTP command handler ordering */
#define FTP_HOOK_FIRST  10
#define FTP_HOOK_MIDDLE 20
#define FTP_HOOK_LAST   30


typedef int ftp_hook_fn(request_rec *r, const char *arg);

/*
 * This is the generic registration function, 
 * see ftp_hook_cmd or ftp_hook_cmd_alias.
 */
FTP_DECLARE(void) ftp_hook_cmd_any(const char *key, ftp_hook_fn *pf, 
                                   const char *alias, int order, 
                                   int flags, const char *help);

/*
 * This this the main hook that is used to hook into mod_ftp.  The
 * key is the command, i.e. "DELE".  The ftp_hook_fn is the function 
 * pointer that will handle this request.  The flags variable is an 
 * integer that is used to tell the dispatcher (ftp_run_cmd) properties
 * about the handler that is about to be run.  These flags include 
 * whether the user needs to be logged in, and how many arguments we 
 * expect the command to have. See the flags above for details.  The 
 * final argument is the help string that will be displayed if the user
 * does "HELP cmd".
 */
#define ftp_hook_cmd(aliascmd, pf, order, flags, desc) \
    ftp_hook_cmd_any((aliascmd), (pf), NULL, (order), \
                     (flags), (desc))

/*
 * This this the alternate hook that is used to hook command aliases
 * into mod_ftp.  The key is the aliased command, i.e. "XPWD".  
 * The ftp_hook_fn is used instead to store the base command, i.e. "PWD".
 * The flags variable is an integer that is used to tell the dispatcher 
 * (ftp_run_cmd) properties about the handler that is about to be run.  
 * These flags must match the base command hook registration.  This macro
 * will always add FTP_ALIAS_CMD to the flags.  See the flags above for 
 * details of other flags.  The final argument is the help string that will 
 * be displayed if the user does "HELP cmd".
 */
#define ftp_hook_cmd_alias(aliascmd, basecmd, order, flags, desc) \
    ftp_hook_cmd_any((aliascmd), NULL, (basecmd), (order), \
                     (flags) , (desc))

/*
 * This is the function that translates ftp aliased methods.
 * It will retreive the base command from the cmd hash.
 */
const char *ftp_get_cmd_alias(const char *key);

/*
 * This is the function that runs the ftp handler.  It will iterate through
 * the linked list of entries that returned from the cmd hash.
 */
int ftp_run_cmd(request_rec *r, const char *key);

/*
 * This function reports (true|false) if a given command should
 * abort an active data transfer
 */
int ftp_cmd_abort_data(const char *key);

typedef struct ftp_cmd_entry ftp_cmd_entry;

/*
 * The FTP command structure contains useful information about the FTP
 * handler.  This information is filled out when a command is registered
 * using ftp_hook_cmd(), which also puts the handler into the global hash.
 */
struct ftp_cmd_entry
{
    const char *key;                  /* The key, e.g. "DELE" */
    ftp_hook_fn *pf;                  /* Pointer to the handler */
    const char *alias;                /* The aliased command e.g. "CDUP" */
    int order;                        /* Handler ordering */
    int flags;                        /* Flags for this command.  See FTP_
                                       * defines above
                                       */
    const char *help;                 /* Help string for this command */
    struct ftp_cmd_entry *next;       /* Pointer to the next handler */
};

#endif
