/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#define FTP_BUILD
#include "mod_ftp.h"

/*
 * callbacks for mod_log_config.
 *
 * These callbacks extend mod_log_config by adding additional
 * % directives as follows:
 *
 * %..M  The mode that was used to transfer the file.
 *       A single character is printed, a (ascii) or b (binary)
 * %..F  Any action that was taken on the file (concationated as needed)
 *         C - file was compressed.
 *         U - file was uncompressed.
 *         T - file was tarred.
 *         _ - no action taken.
 * %..d  Direction the file was sent.
 *         o - outgoing
 *         i - incoming
 * %..W  How the file was accessed.
 *         r - real
 *         a - anonymous
 *         g - guest
 * %..S    Service name, usually 'ftp'
 * %..Z    Authentication method
 *         0 - no auth
 *         1 - rfc931 auth
 * %..Y    Authenticated user id
 *         * - if not available
 */

const char *ftp_log_transfer_mode(request_rec *r, char *a)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    if (fc->type == TYPE_A) {
        return "a";
    }
    else if (fc->type == TYPE_I) {
        return "b";
    }
    else {
        return "-";
    }
}

const char *ftp_log_action_flags(request_rec *r, char *a)
{
    return "_";
}

const char *ftp_log_transfer_direction(request_rec *r, char *a)
{
    if (strcmp(r->method, "RETR") == 0) {
        return "o";
    }
    else if (strcmp(r->method, "STOR") == 0 || strcmp(r->method, "APPE") == 0) {
        return "i";
    }
    else {
        return "-";
    }
}

const char *ftp_log_accessed_anonymously(request_rec *r, char *a)
{
    if (r->user) {
        if (strcasecmp(r->user, "anonymous") == 0) {
            return "a";
        }
        else if (strcasecmp(r->user, "guest") == 0) {
            return "g";
        }
        else {
            return "r";
        }
    }
    else {
        return "n";
    }
}

const char *ftp_log_service_name(request_rec *r, char *a)
{
    return apr_pstrdup(r->pool, "ftp");
}

const char *ftp_log_auth_method(request_rec *r, char *a)
{
    return "0";
}

const char *ftp_log_auth_user_id(request_rec *r, char *a)
{
    return "*";
}
