/* Copyright 2002-2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Apache::Web::WorkerRequest.h managed interface for Apache 2.0 request_rec


#pragma once

#ifndef APACHE_WEB_H
#error "#include \"Apache.Web.h\" ... not WorkerRequest.h!"
#endif

#include "http_header_arrays.h"

using namespace Apache::Web::Helpers;
using namespace System::Collections;

namespace Apache
{
  namespace Web
  {
    // Forward declare a Host
    public __gc class Host;

  public __gc class WorkerRequest : public SimpleWorkerRequest
    {
    private:
        static const char __nogc *response_headers_c __nogc[] = {
            response_headers_m("")
        };
        static const char __nogc *request_headers_c __nogc[] = {
            request_headers_m("")
        };
        static String* response_headers_s[] = {
            response_headers_m(L)
        };
        static String* request_headers_s[] = {
            request_headers_m(L)
        };

        // Create our utf8-encoding to handle utf8->unicode names
        static Encoding *utf8_encoding = new UTF8Encoding(false);
    
        String* uri;
        String* unparsed_uri;
        String* queryargs;
        String* filename;
        String* remoteName;
        String* serverName;

        unsigned char query_raw __gc[];

        String* known_req_header_value[];
        ArrayList* unk_req_header_name;
        ArrayList* unk_req_header_value;
        ArrayList* env_var;
        ArrayList* env_value;

        _gcA_gcA_gcString *unk_req_hdr_arr;

        request_rec __nogc *rr;

        Host* host;
        System::Threading::ManualResetEvent* wait_complete;
        EndOfSendNotification *eosnCallback;
        Object* eosnExtra;

        unsigned char preread_body __gc[];
        int seen_eos;
        int body_sent;
        __int64 bytes_read;

        // Reports an error message in the error log
        void LogRequestError(String *msg, int loglevel, int rv);

    public:
        static bool CodeValidate() {
            if (response_headers_s->get_Length() != ResponseHeaderMaximum)
                return false;
            for (int i = 0; i < ResponseHeaderMaximum; ++i) {
                String *s = HttpWorkerRequest::GetKnownResponseHeaderName(i);
                if (response_headers_s[i]->CompareTo(s) != 0)
                    return false;
            }
            if (request_headers_s->get_Length() != RequestHeaderMaximum)
                return false;
            for (int i = 0; i < RequestHeaderMaximum; ++i) {
                String *s = HttpWorkerRequest::GetKnownRequestHeaderName(i);
                if (request_headers_s[i]->CompareTo(s) != 0)
                    return false;
            }
            return true;
        }

        // Cannot be declared inline, we don't have Host defined
        virtual String* GetAppPath(void); 
        virtual String* GetAppPathTranslated(void); 

        WorkerRequest(int Req, Host *ThisHost);

        int Handler(void);
        [ComVisible(false)]
        virtual void SetEndOfSendNotification(
                         HttpWorkerRequest::EndOfSendNotification *callback,
                         Object* extraData) 
        {
#ifdef _DEBUG
            LogRequestError(L"SetEndOfSendNotification: initialized", APLOG_DEBUG, 0);
#endif
        }

        virtual void EndOfRequest(void) 
        {
            // If ProcessRequest() in Handler completed before the request
            // request processing completes, it blocks on wait_complete, 
            // we need to toggle the event that Handler() is blocked on
            // or would block on upon return from ProcessRequest().
            if (eosnCallback) {
#ifdef _DEBUG
                LogRequestError(L"EndOfRequest: invoked EndOfSendNotification", APLOG_DEBUG, 0);
#endif
                eosnCallback->Invoke(this, eosnExtra);
                eosnCallback = NULL;
            }
#ifdef _DEBUG
            else
                LogRequestError(L"EndOfRequest: processed", APLOG_DEBUG, 0);
#endif
            wait_complete->Set();
        }

        virtual String* GetUriPath(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetUriPath: returns ", uri);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return uri;
        }

        virtual String* GetQueryString(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetQueryString: returns ", queryargs);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return queryargs;
        }

        virtual unsigned char GetQueryStringRawBytes(void) __gc[]
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetQueryStringRawBytes: returns ",
                                         (new Int32(query_raw->get_Length()))->ToString(),
                                         L" raw characters");
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return query_raw;
        }

        virtual String* GetRawUrl(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetRawUrl: returns ", unparsed_uri);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return unparsed_uri;
        }

        virtual String* GetHttpVerbName(void) 
        {
            String* verb = new String(rr->method);
#ifdef _DEBUG
            String *res = String::Concat(L"GetHttpVerbName: returns ", verb);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return verb;
        }

        virtual String* GetHttpVersion(void) 
        {
            String* version = new String(rr->protocol);
#ifdef _DEBUG
            String *res = String::Concat(L"GetHttpVersion: returns ", version);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return version;
        }

        virtual String* GetProtocol(void)
        {
            // TODO: HTTPS when appropriate
#ifdef _DEBUG
            LogRequestError(L"GetProtocol: returns HTTP", APLOG_DEBUG, 0);
#endif
            return new String(L"HTTP");
        }

        virtual String* GetRemoteAddress(void) 
        {
            String* addr = new String(rr->connection->remote_ip);
#ifdef _DEBUG
            String *res = String::Concat(L"GetRemoteAddress: returns ", addr);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return addr;
        }

        virtual int GetRemotePort(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetRemotePort: returns ", 
                (new Int32(rr->connection->remote_addr->port))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
	    return rr->connection->remote_addr->port;
        }

        virtual String* GetRemoteName(void)
        {
            if (!remoteName) {
                const char __nogc *name = ap_get_remote_host(rr->connection, 
                                                            rr->per_dir_config, 
                                                            REMOTE_HOST, NULL);
                if (!name)
                    remoteName = String::Empty;
                else
                    remoteName = new String(name);
            }
#ifdef _DEBUG
            String *res = String::Concat(L"GetRemoteName: returns ", remoteName);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return remoteName;
        }

        virtual String* GetServerName(void)
        {
            if (!serverName) {
                const char __nogc *name = ap_get_server_name(rr);
                if (!name)
                    serverName = String::Empty;
                else
                    serverName = new String(name);
            }
#ifdef _DEBUG
            String *res = String::Concat(L"GetServerName: returns ", serverName);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return serverName;
        }

        virtual String* GetServerVariable(String *name)
        {
            if (name->Equals(L"ALL_RAW")) {
                int size = 1;
                for (int i = 0; i < env_var->get_Count(); ++i) {
                    size += static_cast<String*>(env_var->get_Item(i))->get_Length() + 1;
                    size += static_cast<String*>(env_value->get_Item(i))->get_Length() + 1;
                }
                StringBuilder *res = new StringBuilder(size);
                for (int i = 0; i < env_var->get_Count(); ++i) {
                    res = res->Append(static_cast<String*>(env_var->get_Item(i)));
                    res = res->Append(L"=");
                    res = res->Append(static_cast<String*>(env_value->get_Item(i)));
                    res = res->Append(L"\0", 0, 1);
                }
                res = res->Append(L"\0", 0, 1);
#ifdef _DEBUG
                LogRequestError(L"GetServerVariable: for ALL_RAW returns all", APLOG_DEBUG, 0);
#endif
                return res->ToString();
            }
            int ent = env_var->IndexOf(name);
            if (ent >= 0) {
#ifdef _DEBUG
            String *res = String::Concat(L"GetServerVariable: for ", name, L" returns ", 
                                         static_cast<String*>(env_value->get_Item(ent)));
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
                return static_cast<String*>(env_value->get_Item(ent));
            }
#ifdef _DEBUG
            String *res = String::Concat(L"GetServerVariable: for ", name,
                                         L" returns NULL (not found)");
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return NULL;
        }

        virtual String* GetLocalAddress(void) 
        {
            String* addr = new String(rr->connection->local_ip);
#ifdef _DEBUG
            String *res = String::Concat(L"GetLocalAddress: returns ", addr);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return addr;
        }

        virtual int GetLocalPort(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetLocalPort: returns ", 
                (new Int32(rr->connection->local_addr->port))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return rr->connection->local_addr->port;
        }

        virtual String* GetFilePath(void); 

        virtual String* GetFilePathTranslated(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetFilePathTranslated: returns ", filename);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return filename;
        }

        virtual String* GetPathInfo(void) 
        {
            String* path_info = new String(rr->path_info);
#ifdef _DEBUG
            String *res = String::Concat(L"GetPathInfo: returns ", path_info);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return path_info;
        }

        virtual String* GetKnownRequestHeader(int index)
        {
            if (index >= 0 && index < RequestHeaderMaximum) {
#ifdef _DEBUG
                String *res = String::Concat(L"GetKnownRequestHeader: for ", 
                                             request_headers_s[index],
                                             L" returns ", 
                                             known_req_header_value[index]);
                LogRequestError(res, APLOG_DEBUG, 0);
#endif
                return known_req_header_value[index];
            }
#ifdef _DEBUG
            String *res = String::Concat(L"GetKnownRequestHeader: for ", 
                                         (new Int32(index))->ToString(),
                                         L" returns NULL (not found)");
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return NULL;
        }
    
        virtual String* GetUnknownRequestHeader(String* name) 
        {
            int ent = unk_req_header_name->IndexOf(name);
            if (ent >= 0) {
                String *val = static_cast<String*>(unk_req_header_value->get_Item(ent));
#ifdef _DEBUG
                String *res = String::Concat(L"GetUnknownRequestHeader: for ", name,
                                             L" returns ", val);
                LogRequestError(res, APLOG_DEBUG, 0);
#endif
                return val;
            }
#ifdef _DEBUG
            String *res = String::Concat(L"GetUnknownRequestHeader: for ", name,
                                         L" returns NULL (not found)");
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return NULL;
        }

        [System::CLSCompliantAttribute(false)]
        virtual String __gc* GetUnknownRequestHeaders(void) __gc[] __gc[]
        {
#ifdef _DEBUG
            LogRequestError(L"GetUnknownRequestHeaders: returns all", APLOG_DEBUG, 0);
#endif
            return unk_req_hdr_arr->Value();
        }

        static int GetKnownRequestHeaderIndex(String *header)
        {
            for (int i = 0; i < RequestHeaderMaximum; ++i) {
                if (String::Compare(header, request_headers_s[i]) == 0) {
                    return i;
                }
            }
            return -1;
        }

        static String *GetKnownRequestHeaderName(int header)
        {
            if (header >= 0 && header < RequestHeaderMaximum) {
                return request_headers_s[header];
            }
            return NULL;
        }

        static int GetKnownResponseHeaderIndex(String *header)
        {
            for (int i = 0; i < ResponseHeaderMaximum; ++i) {
                if (String::Compare(header, response_headers_s[i]) == 0)
                    return i;
            }
            return -1;
        }

        static String *GetKnownResponseHeaderName(int header)
        {
            if (header >= 0 && header < ResponseHeaderMaximum) {
                return response_headers_s[header];
            }
            return NULL;
        }

        virtual bool HasEntityBody(void)
        {
#ifdef _DEBUG
            String *res = String::Concat(L"HasEntityBody: returns ", 
                (new Boolean(!seen_eos || bytes_read))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return !seen_eos || bytes_read;
        }

        virtual int ReadEntityBody(unsigned char buffer __gc[], int size);

        virtual __int64 GetBytesRead(void)
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetBytesRead: returns ", 
                                         (new Int64(bytes_read))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif            
            return bytes_read;
        }

        virtual unsigned char GetPreloadedEntityBody(void) __gc[]
        {
#ifdef _DEBUG
            String *res = String::Concat(L"GetPreloadedEntityBody: returns ", 
                (new Int32(preread_body->get_Length()))->ToString(), 
                L" bytes");
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return preread_body;
        }

        virtual bool IsEntireEntityBodyIsPreloaded(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"IsEntireEntityBodyIsPreloaded: returns ", 
                (new Boolean(seen_eos != 0))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return seen_eos != 0;
        }

        virtual bool IsClientConnected(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"IsClientConnected: returns ", 
                (new Boolean(!rr->connection->aborted))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return !rr->connection->aborted;
        }

        virtual bool HeadersSent(void) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"HeadersSent: returns ", 
                (new Boolean(body_sent != 0))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            return (body_sent != 0);
        }

        virtual void SendStatus(int statusCode, String* statusDescription) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"SendStatus: sending ", 
                                         (new Int32(statusCode))->ToString(), 
                                         L" with desc ", statusDescription);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            rr->status = statusCode;
            if (!statusDescription 
                    || statusDescription->Equals(String::Empty)) {
                rr->status_line = ap_get_status_line(rr->status);
            }
            else {
                unsigned char avalue __gc[] 
                    = utf8_encoding->GetBytes(statusDescription);
                unsigned char __pin *status_desc = &avalue[0];
                char code __nogc[12];
                itoa(rr->status, code, 10);
                rr->status_line = apr_pstrcat(rr->pool, code, " ", 
                                              (char __nogc*)status_desc, NULL);
            }
        }
        
        virtual void SendKnownResponseHeader(int index, String* value)
        {
            if (index >= 0 && index < ResponseHeaderMaximum) {
#ifdef _DEBUG
                String *res = String::Concat(L"SendKnownResponseHeader: ", 
                                             response_headers_c[index], 
                                             L" as ", value);
                LogRequestError(res, APLOG_DEBUG, 0);
#endif
                unsigned char avalue __gc[] = utf8_encoding->GetBytes(value);
                int len = avalue->Count;
                unsigned char __pin *cvalue = &avalue[0];
                char __nogc *poolval = static_cast<char __nogc *>
                                          (apr_pmemdup(rr->pool, cvalue, len + 1));
                poolval[len] = '\0';
                if (index == HeaderContentType)
                    ap_set_content_type(rr, poolval);
                else
                    apr_table_setn(rr->headers_out, response_headers_c[index], poolval);
            }
#ifdef _DEBUG
            else {
                String *res = String::Concat(L"SendKnownResponseHeader: failed to send ", 
                                            (new Int32(index))->ToString(), 
                                            L" as ", value);
                LogRequestError(res, APLOG_DEBUG, 0);
            }
#endif
        }

        virtual void SendUnknownResponseHeader(String* name, String* value) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"SendUnknownResponseHeader: ", name, 
                                         L" as ", value);
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            unsigned char aname __gc[] = utf8_encoding->GetBytes(name);
            int len = aname->Count;
            unsigned char __pin *cname = &aname[0];
            char __nogc *poolkey = static_cast<char __nogc *>
                                        (apr_pmemdup(rr->pool, cname, len + 1));
            poolkey[len] = '\0';
            unsigned char avalue __gc[] = utf8_encoding->GetBytes(value);
            len = avalue->Count;
            unsigned char __pin *cvalue = &avalue[0];
            char __nogc *poolval = static_cast<char __nogc *>
                                        (apr_pmemdup(rr->pool, cvalue, len + 1));
            poolval[len] = '\0';
            apr_table_setn(rr->headers_out, poolkey, poolval);
        }

        virtual void SendCalculatedContentLength(int contentLength) 
        {
#ifdef _DEBUG
            String *res = String::Concat(L"SendCalculatedContentLength: ", 
                                        (new Int32(contentLength))->ToString());
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            ap_set_content_length(rr, contentLength);
        }

        virtual void SendResponseFromMemory(IntPtr data, int length);

        virtual void SendResponseFromMemory(unsigned char data __gc[], __int32 length) 
        {
            if (!length) {
#ifdef _DEBUG
                LogRequestError("SendResponseFromMemory: passed zero byte buffer!", APLOG_DEBUG, 0);
#endif
                return;
            }
#ifdef _DEBUG
            LogRequestError(L"SendResponseFromFile: pinning GC buffer", APLOG_DEBUG, 0);
#endif
            unsigned char __pin *data_buf = &data[0];
            IntPtr data_ptr = static_cast<void*>(data_buf);
            SendResponseFromMemory(data_ptr, length);
        }

        virtual void SendResponseFromFile(IntPtr handle, __int64 offset, __int64 length);

        virtual void SendResponseFromFile(String* filename, __int64 offset, __int64 length) 
        {
            if (!length) {
#ifdef _DEBUG
                LogRequestError("SendResponseFromFile: passed zero byte length!", APLOG_DEBUG, 0);
#endif
                return;
            }
#ifdef _DEBUG
            String *res = String::Concat(L"SendResponseFromFile: opening ", 
                                         filename); 
            LogRequestError(res, APLOG_DEBUG, 0);
#endif
            FileStream *stream = new FileStream(filename, FileMode::Open, 
                                                FileAccess::Read, 
                                                FileShare::Read, 0, true);
            SendResponseFromFile(stream->Handle, offset, length);
            stream->Close();
        }

        virtual void FlushResponse(bool finalFlush);

        virtual void CloseConnection(void)
        {
#ifdef _DEBUG
            LogRequestError(L"CloseConnection: requested (noop)", APLOG_DEBUG, 0);
#endif
            ; // Noop, we trust Apache's determination here of the protocol
        }

        virtual String* MapPath(String* path);
    };
  }
}
