/* Copyright 2002-2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// mod_aspdotnet.so - Microsoft ASP.NET connector module for Apache 2.0


// We don't care about unused arguments
#pragma warning(disable: 4100)

// We require much Winfoo to intermix Com/Ole/CLR code...
// don't wait for APR to include a stripped down Win32 API
//
#define WIN32_LEAN_AND_MEAN
#define _WIN32_WINNT 0x0400
#include <Windows.h>
#include <winsock2.h>
#include <objbase.h>

#include "apr.h"
#include "apr_lib.h"
#include "apr_strings.h"
#include "apr_buckets.h"
#include "apr_pools.h"

#include "httpd.h"
#include "http_core.h"
#include "http_config.h"
#include "http_log.h"
#include "http_main.h"
#include "http_request.h"
#include "util_filter.h"
#include "util_script.h"

#include "Apache.Web.Version.h"

#include <mscoree.h>
#import <mscorlib.tlb> raw_interfaces_only \
    high_property_prefixes("_get","_put","_putref") \
    rename("ReportEvent", "CorReportEvent")
using namespace mscorlib;

#import "Apache.Web.tlb"
using namespace Apache_Web;

extern "C" {
    extern module AP_MODULE_DECLARE_DATA aspdotnet_module;
}

// Initialized or recovered (from userdata) in pre_config
typedef struct asp_net_host_conf_t 
{
    // Allocation issue; virtual and physical are hashed
    // as a combined string of /vpath\0/ppath\0 with the
    // combined_len including both NULLs.  This permits us
    // to quickly find given virtual/physical mappings for
    // given vhosts (they are bound to a specific virutal
    // and physical path pair.)
    apr_size_t  combined_len;
    const char *virtual_path;
    const char *physical_path;
    int         host_key;
//    IApacheWebHost *pHost;
} asp_net_host_conf_t;

struct asp_net_conf_t {
    apr_pool_t *pool;
    apr_hash_t *hosts;  // Hash of asp_net_host_conf_t records
    ICorRuntimeHost *pCorRuntime;
    IApacheWebHostFactory *pHostFactory;
    HANDLE lock_module;
} *conf;

// Initialized for each restart
typedef struct asp_net_alias_t {
    // These are hashed per-vhost only by the virtual_path.
    apr_size_t  virtual_path_len;
    const char *virtual_path;
    asp_net_host_conf_t *host;
} asp_net_alias_t;

typedef struct asp_net_sconf_t {
    apr_hash_t *aliases;  // Hash of asp_net_alias_t records
} asp_net_sconf_t;


static void *create_asp_net_sconf(apr_pool_t *p, server_rec *s)
{
    asp_net_sconf_t *sconf;
    sconf = (asp_net_sconf_t*)apr_palloc(p, sizeof(asp_net_sconf_t));
    sconf->aliases = apr_hash_make(p);
    return sconf;
}

static void *merge_asp_net_sconfs(apr_pool_t *p, void *basev, void *addv)
{
    asp_net_sconf_t *base = (asp_net_sconf_t*)basev;
    asp_net_sconf_t *add = (asp_net_sconf_t*)addv;
    apr_hash_overlay(p, add->aliases, base->aliases);
    return add;
}


static const char *asp_net_mount(cmd_parms *cmd, void *dummy, 
                                 const char *uri, const char *rootarg)
{
    char *root;
    asp_net_sconf_t *sconf 
        = (asp_net_sconf_t*)ap_get_module_config(cmd->server->module_config,
                                                 &aspdotnet_module);
    asp_net_alias_t *mount 
        = (asp_net_alias_t*)apr_palloc(cmd->pool, sizeof(*mount));

    mount->virtual_path_len = strlen(uri);

    apr_status_t rv = apr_filepath_merge((char**)&root, ap_server_root, rootarg,
                                         APR_FILEPATH_TRUENAME, cmd->pool);

    if (rv != APR_SUCCESS || !root) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, NULL,
                     "mod_aspdotnet: Failed to resolve the full file path"
                     "for AspNetMount \"%s\" \"%s\"", uri, rootarg);
        return NULL;
    }

    apr_finfo_t fi;
    rv = apr_stat(&fi, root, APR_FINFO_TYPE, cmd->temp_pool);
    if ((rv != APR_SUCCESS) || (fi.filetype != APR_DIR)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, NULL,
            "mod_aspdotnet: File path is a directory, or could not stat path "
                     "for AspNetMount \"%s\" \"%s\"", uri, rootarg);
        return NULL;
    }

    apr_size_t root_len     = strlen(root);
    apr_size_t combined_len = mount->virtual_path_len + root_len + 2;
    mount->virtual_path = (const char*)apr_palloc(cmd->pool, combined_len);
    memcpy((char*)mount->virtual_path, uri, mount->virtual_path_len + 1);
    memcpy((char*)mount->virtual_path + mount->virtual_path_len + 1, 
           root, root_len + 1);

    // Recover the host matching this virtual+physical path combo, 
    // or create a new persistant host record (on the first pass.)
    mount->host = (asp_net_host_conf_t*)apr_hash_get(conf->hosts, 
                                                     mount->virtual_path,
                                                     combined_len);
    if (!mount->host) {
        mount->host = (asp_net_host_conf_t*)apr_palloc(conf->pool, 
                                                       sizeof(*mount->host));
        mount->host->combined_len = combined_len;
        mount->host->virtual_path = (const char*)apr_palloc(conf->pool, 
                                                            combined_len);
        mount->host->physical_path = mount->host->virtual_path 
                                   + mount->virtual_path_len + 1;
        memcpy((char*)mount->host->virtual_path, 
               mount->virtual_path, combined_len);
        mount->host->host_key = -1;
        apr_hash_set(conf->hosts, mount->host->virtual_path, 
                     mount->host->combined_len, mount->host);
    }

    apr_hash_set(sconf->aliases, mount->virtual_path, 
                 mount->virtual_path_len, mount);
    return NULL;
}

static const command_rec asp_net_cmds[] =
{
    AP_INIT_TAKE2("AspNetMount", (cmd_func)asp_net_mount, NULL, RSRC_CONF,
                  "Mount a URI path to an Asp.Net AppDomain path root"),
    {NULL}
};

static apr_status_t asp_net_stop(void *dummy)
{
    conf->pHostFactory->Destroy();

    if (conf->pHostFactory) {
        conf->pHostFactory->Release();
    }
    if (conf->pCorRuntime) {
        conf->pCorRuntime->Stop();
        conf->pCorRuntime->Release();
    }
    return APR_SUCCESS;
}

static HRESULT init_asp_engine(void)
{
    HRESULT hr;

    // Get the path to this mod_aspdotnet module so that we can
    // lock ourselves in-place for the lifetime of the server
    wchar_t wAspNetPath[APR_PATH_MAX];
    HMODULE hModule = GetModuleHandleW(L"mod_aspdotnet.so");
    if (!GetModuleFileNameW(hModule, wAspNetPath, APR_PATH_MAX)
            || (wcslen(wAspNetPath) >= APR_PATH_MAX - 1)) {
        hr = GetLastError();
        apr_status_t rv = APR_FROM_OS_ERROR(hr);
        if (!rv) {
            hr = E_FAIL;
            rv = APR_ENAMETOOLONG;
        }
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, NULL,
                     "mod_aspdotnet: Failed to discover the full file path to "
                     "mod_aspdotnet.so.  (Was it renamed?)");
        _com_raise_error(hr);
    }
    conf->lock_module = LoadLibraryW(wAspNetPath);

    // Now we are prepared to register our cleanup in the global
    // process pool, because we trust the module cannot be unloaded
    // by apr_dso_unload [the module instance is refcounted]
    apr_pool_cleanup_register(conf->pool, NULL, asp_net_stop, 
                              apr_pool_cleanup_null);

    // Now get the path to the apache.exe binary, as Apache.Web
    // reside in the same path as the server's dll libraries.
    if (!GetModuleFileNameW(NULL, wAspNetPath, APR_PATH_MAX)
            || (wcslen(wAspNetPath) >= APR_PATH_MAX - 17)) {
        hr = GetLastError();
        apr_status_t rv = APR_FROM_OS_ERROR(hr);
        if (!rv) {
            hr = E_FAIL;
            rv = APR_ENAMETOOLONG;
        }
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, NULL,
                     "mod_aspdotnet: Unable to determine apache.exe path!");
        _com_raise_error(hr);
    }

    wchar_t *repl = wcsrchr(wAspNetPath, L'\\');
    if (!repl) {
        ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, NULL,
                     "mod_aspdotnet: Unable to determine apache.exe path!");
        _com_raise_error(E_FAIL);
    }

    ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, NULL,
                 "mod_aspdotnet: Module initialization commencing...");

    DWORD len;
    hr = GetCORVersion(wAspNetPath, sizeof(wAspNetPath) 
                                     / sizeof(WCHAR) - 1, &len);
    if (FAILED(hr)) {
        wAspNetPath[len] = L'\0';
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: GetCORVersion failed to return "
                     "the .NET CLR engine version.");
        _com_raise_error(hr);
    }

    hr = CorBindToRuntimeEx(wAspNetPath, 
                            L"svr",  // Or "wks" 
                            STARTUP_LOADER_OPTIMIZATION_MULTI_DOMAIN_HOST | 
                            STARTUP_CONCURRENT_GC, 
                            CLSID_CorRuntimeHost, 
                            IID_ICorRuntimeHost, 
                            (void **)&conf->pCorRuntime);
    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not CorBindToRuntimeEx "
                     "for the .NET CLR engine.");
        _com_raise_error(hr);
    }

    hr = conf->pCorRuntime->Start();
    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not start the "
                     ".NET CLR engine.");
        _com_raise_error(hr);
    }

    ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, NULL,
                 "mod_aspdotnet: Module started .NET CLR...");

    IUnknown *pAppDomainIUnk = NULL;
    hr = conf->pCorRuntime->GetDefaultDomain(&pAppDomainIUnk);
    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the .NET default "
                     "application domain.");
        _com_raise_error(hr);
    }
    if (!pAppDomainIUnk) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the .NET default "
                     "application domain's interface.");
        _com_raise_error(E_NOINTERFACE);
    }

    _AppDomain *pDefaultDomain = NULL;
    hr = pAppDomainIUnk->QueryInterface(__uuidof(_AppDomain), 
                                        (void**)&pDefaultDomain);
    // Done with pAppDomainIUnk
    pAppDomainIUnk->Release();

    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the .NET default "
                     "application domain's _AppDomain interface.");
        _com_raise_error(hr);
    }
    if (!pDefaultDomain) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the .NET default "
                     "application domain _AppDomain interface.");
        _com_raise_error(E_NOINTERFACE);
    }

    ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, NULL,
                 "mod_aspdotnet: Module initialized .NET default AppDomain...");

    _ObjectHandle *pObjHandle = NULL;
    hr = pDefaultDomain->CreateInstance(_bstr_t("Apache.Web, "
                                            "Version=" APACHE_WEB_VERSION ", "
				            "Culture=neutral, "
                                            "PublicKeyToken=9b9b842f49b86351"),
                                        _bstr_t(L"Apache.Web.HostFactory"),
                                        &pObjHandle);
    // Done with pDefaultDomain
    pDefaultDomain->Release();

    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not create the .NET interface "
                     "for the Apache.Web.HostFactory.");
        _com_raise_error(hr);
    }
    if (!pObjHandle) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the .NET object "
                     "for the Apache.Web.HostFactory.");
        _com_raise_error(E_NOINTERFACE);
    }
    
    VARIANT vHostFactory;
    VariantInit(&vHostFactory);
    hr = pObjHandle->Unwrap(&vHostFactory);
    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not unwrap the COM interface "
                     "for the Apache.Web.HostFactory.");
        _com_raise_error(hr);
    }
    if (!vHostFactory.pdispVal) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the unknown COM "
                     "interface for the Apache.Web.HostFactory.");
        _com_raise_error(E_NOINTERFACE);
    }

    hr = vHostFactory.pdispVal->QueryInterface(__uuidof(IApacheWebHostFactory),
                                               (void**)&conf->pHostFactory);
    // Done with vHostFactory
    VariantClear(&vHostFactory);

    if (FAILED(hr)) { 
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not retrieve the COM interface "
                     "for the Apache.Web.HostFactory.");
        _com_raise_error(hr);
    }

    ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, NULL,
                 "mod_aspdotnet: Module initialized HostFactory...");

    // Test invocation, assure we have a good hostfactory
    hr = conf->pHostFactory->Configure(L"");
    if (FAILED(hr)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), NULL,
                     "mod_aspdotnet: Could not correctly configure the "
                     "Apache.Web.HostFactory.");
        _com_raise_error(hr);
    }

    ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, NULL,
                 "mod_aspdotnet: HostFactory configuration complete.");

    return S_OK;
}

static int asp_net_handler(request_rec *r)
{
    if (!r->handler || strcmp(r->handler, "asp.net") != 0) {
        return DECLINED;
    }

    if (r->finfo.filetype != APR_REG) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, r, 
                "mod_aspdotnet: File not found or unable to stat: %s",
                r->filename);
        return HTTP_NOT_FOUND;
    }

    if (!r->uri || (r->uri[0] != '/')) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, r, 
                "mod_aspdotnet: Forbidden, request URI was not absolute: %s",
                r->uri ? r->uri : "<NULL>");
        return HTTP_FORBIDDEN;
    }

    if (!(ap_allow_options(r) & OPT_EXECCGI)) 
    {
        ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, r, 
                "mod_aspdotnet: Forbidden, Options ExecCGI is not set for: %s",
                r->filename);
        return HTTP_FORBIDDEN;
    }

    if ((r->used_path_info == AP_REQ_REJECT_PATH_INFO) &&
        r->path_info && *r->path_info)
    {
        /* default to accept */
            ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, r, 
            "mod_aspdotnet: AcceptPathInfo off for %s disallows user's path %s",
            r->filename, r->path_info);
        return HTTP_NOT_FOUND;
    }

    asp_net_sconf_t *sconf 
        = (asp_net_sconf_t*)ap_get_module_config(r->server->module_config, 
                                                 &aspdotnet_module);

    asp_net_host_conf_t *host = NULL;
    apr_hash_index_t *item;
    for (item = apr_hash_first(r->pool, sconf->aliases); item; 
            item = apr_hash_next(item)) 
    {
        asp_net_alias_t *alias;
        apr_hash_this(item, NULL, NULL, (void**)&alias);
        if (strncasecmp(alias->virtual_path, r->uri, 
                        alias->virtual_path_len) == 0) {
            host = alias->host;
            break;
        }
    }

    if (!item || !host) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, r,
                "mod_aspdotnet: No AspNetMount URI for request: %s",
                r->uri);
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    // We never create an environment, however, we will query the
    // r->env to recover these values.
    ap_add_common_vars(r);
    ap_add_cgi_vars(r);

    // Check that THIS thread is coinitialized.
    HRESULT hr = CoInitialize(NULL);
    if (hr == RPC_E_CHANGED_MODE)
        hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);

    if (FAILED(hr)) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), r,
                      "mod_aspdotnet: CoInitialize failed!");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    VARIANT vRequest;
    VariantInit(&vRequest);

    int status;

    try {
        status = conf->pHostFactory->HandleHostRequest(host->host_key, 
                                                       (UINT_PTR)r);
    }
    catch (_com_error err) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(err.Error()), r,
                    "mod_aspdotnet: CreateRequest %s processing failed, file %s",
                    r->uri, r->filename);
        char *desc = (char*)err.Description();
        if (desc) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, 
                          APR_FROM_OS_ERROR(err.Error()), r,
                          "mod_aspdotnet: %s", desc);
        }
        status = HTTP_INTERNAL_SERVER_ERROR;
    }
    catch (HRESULT hr) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), r,
                      "mod_aspdotnet: CreateRequest %s processing failed, file %s",
                      r->uri, r->filename);
        status = HTTP_INTERNAL_SERVER_ERROR;
    }
    catch (...) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, r,
                      "mod_aspdotnet: CreateRequest %s processing failed, file %s",
                      r->uri, r->filename);
        status = HTTP_INTERNAL_SERVER_ERROR;
    }

    VariantClear(&vRequest);

    return status;
}

static int asp_net_pre_config(apr_pool_t *pconf, apr_pool_t *plog, 
                              apr_pool_t *ptemp)
{
    apr_pool_t *process = apr_pool_get_parent(pconf);

    // Recover or create the global, persistant configuration
    // Setup and teardown can take a considerable amount of time,
    // we do not want to repeat this twice per Apache process.
    if ((apr_pool_userdata_get((void**)&conf, "mod_aspdotnet::global_conf", 
                               process) != APR_SUCCESS) || !conf) {
        HRESULT hr = CoInitialize(NULL);
        if (hr == RPC_E_CHANGED_MODE)
            hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
        if (FAILED(hr)) {
            ap_log_error(APLOG_MARK, APLOG_CRIT, APR_FROM_OS_ERROR(hr), NULL,
                        "mod_aspdotnet: Failed to CoInitialize the threaded "
                        "COM engine for .NET CLR interop!");
            return 1;
        }
        conf = (asp_net_conf_t*)apr_palloc(process, sizeof(*conf));
        conf->pool  = process;
        conf->hosts = apr_hash_make(process);
        conf->pCorRuntime  = NULL;
        conf->pHostFactory = NULL;
        apr_pool_userdata_setn(conf, "mod_aspdotnet::global_conf", 
                               apr_pool_cleanup_null, process);
    }
    return APR_SUCCESS;
}


static int asp_net_post_config(apr_pool_t *pconf, apr_pool_t *plog, 
                               apr_pool_t *ptemp, server_rec *gs)
{
    apr_hash_index_t *item;

    // First time through, initialize .Net and the HostFactory
    if (!conf->pCorRuntime || !conf->pHostFactory) {
        try {
            init_asp_engine();
        }
        catch (_com_error err) {
            char *desc = (char*)err.Description();
            ap_log_error(APLOG_MARK, APLOG_CRIT, 
                         APR_FROM_OS_ERROR(err.Error()), gs,
                         "mod_aspdotnet: Failed to start Asp.Net "
                         "Apache.Web host factory");
            if (desc) {
                ap_log_error(APLOG_MARK, APLOG_CRIT, 
                            APR_FROM_OS_ERROR(err.Error()), gs,
                            "mod_aspdotnet: %s", desc);
            }
            return 1;
        }
        catch (HRESULT hr) {
            ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), gs,
                         "mod_aspdotnet: Failed to start Asp.Net "
                         "Apache.Web host factory.");
            return 1;
        }
        catch (...) {
            ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, gs,
                         "mod_aspdotnet: Failed to start Asp.Net "
                         "Apache.Web host factory.");
            return 1;
        }
    }

    for (item = apr_hash_first(ptemp, conf->hosts); item; 
            item = apr_hash_next(item)) 
    {
        asp_net_host_conf_t *host;
        apr_hash_this(item, NULL, NULL, (void**)&host);
    
        if (host->host_key != -1) {
            continue;
        }

        try {
            // XXX: i18n these paths by treating them as UTF-8 -> Unicode!!!
            host->host_key = conf->pHostFactory->CreateHost(
                                                 _bstr_t(host->virtual_path), 
                                                 _bstr_t(host->physical_path),
                                                 (int) gs);
            if (host->host_key == -1)
                _com_raise_error(E_NOINTERFACE);
        }
        catch (_com_error err) {
            ap_log_error(APLOG_MARK, APLOG_CRIT, 
                         APR_FROM_OS_ERROR(err.Error()), gs, 
                         "mod_aspdotnet: Failed to create Host connector "
                         "for %s mapped to %s", host->virtual_path, 
                         host->physical_path);
            ap_log_error(APLOG_MARK, APLOG_CRIT, 
                         APR_FROM_OS_ERROR(err.Error()), gs, 
                         "mod_aspdotnet: %s", (char*)err.Description());
            return 1;
        }
        catch (HRESULT hr) {
            ap_log_error(APLOG_MARK, APLOG_ERR, APR_FROM_OS_ERROR(hr), gs,
                         "mod_aspdotnet: Failed to create Host of for %s mapped "
                         "to %s", host->virtual_path, host->physical_path);
            return 1;
        }
        catch (...) {
            ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, 0, gs,
                         "mod_aspdotnet: Failed to create Host of for %s mapped "
                         "to %s", host->virtual_path, host->physical_path);
            return 1;
        }
    }

    return OK;
}

static void register_hooks(apr_pool_t *p)
{
    ap_hook_handler(asp_net_handler, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_pre_config(asp_net_pre_config, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_post_config(asp_net_post_config, NULL, NULL, APR_HOOK_MIDDLE);
}

module AP_MODULE_DECLARE_DATA ::aspdotnet_module =
{
    STANDARD20_MODULE_STUFF,
    NULL,                       /* dir config */
    NULL,                       /* merge dir config */
    create_asp_net_sconf,       /* server config */
    merge_asp_net_sconfs,       /* merge server config */
    asp_net_cmds,               /* command apr_table_t */
    register_hooks              /* register hooks */
};
