/* Copyright 2002-2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Apache::Web::HostFactory.h managed host engine for mod_aspdotnet initialization


#pragma once

#ifndef APACHE_WEB_H
#error "#include \"Apache.Web.h\" ... not HostFactory.h!"
#endif

namespace Apache
{
  namespace Web
  {
    // Forward declare a Host
    public __gc class Host;

    // The HostFactory engine to instantiate new Host objects
    public __gc class HostFactory 
        : public IApacheWebHostFactory, public MarshalByRefObject
    {
    private:
        // Create our utf8-encoding to handle utf8->unicode names
        Encoding* utf8_encoding;
        ArrayList* hostURI;
        ArrayList* hostDir;
        ArrayList* hostObj;
        ArrayList* hostSvr;

    private:
        // Reports an error message in the error log
        void LogServerError(String *msg, int loglevel, int rv, server_rec __nogc *s)
        {
            unsigned char avalue __gc[] = utf8_encoding->GetBytes(msg);
            unsigned char __pin *msg_text = &avalue[0];
            ap_log_error("Apache.Web", 0, loglevel|(rv ? 0 : APLOG_NOERRNO),
                         rv, s, "mod_aspdotnet: %s", (char*) msg_text, NULL);
        }


    public:
        HostFactory() {
            utf8_encoding = new UTF8Encoding(false);
            hostURI = new ArrayList();
            hostDir = new ArrayList();
            hostObj = new ArrayList();
            hostSvr = new ArrayList();
        }

        virtual Object* InitializeLifetimeService()
        {
            // Indestructable, for good reason
            return NULL;
        }

        // Returns E_FAIL if HostFactory cannot be configured; path is a noop argument
        HRESULT Configure(String *path) 
        {
#ifdef _DEBUG
            if (!WorkerRequest::CodeValidate())
                return E_FAIL;
            else
#endif
                return S_OK;
        }

        void Destroy(void)
        {
            if (utf8_encoding) {
                utf8_encoding = NULL;
                int hosts = hostObj->get_Count();
                for (int i = 0; i < hosts; ++i)
                {
                    // Don't fault if the hostObj is already destroyed
                    try {
                        hostObj->set_Item(i, NULL);
                    }
                    catch (Exception * /*e*/) {
                    }
                }
                hostURI = NULL;
                hostDir = NULL;
                hostObj = NULL;
                hostSvr = NULL;
            }
        }

        // Instantiate a new Host object; trailing slashes are acceptable
        bool ConnectHost(int HostKey) 
        {
            Type* hostType = __typeof(Host);
            Object* newHost;

            try {
                newHost = ApplicationHost::CreateApplicationHost(hostType,
                             __try_cast<String*>(hostURI->get_Item(HostKey)),
                             __try_cast<String*>(hostDir->get_Item(HostKey)));
                hostObj->set_Item(HostKey, newHost);
                                  
                if (newHost) {
                    Host *host = __try_cast<Host*>(newHost);
                    host->Configure(
                             __try_cast<String*>(hostURI->get_Item(HostKey)),
                             __try_cast<String*>(hostDir->get_Item(HostKey)),
                             this, HostKey);
                    return true;
                }
                else {
                    server_rec __nogc *s = (server_rec __nogc *)
                        (*__try_cast<__box int*>(hostSvr->get_Item(HostKey)));
                    LogServerError(L"mod_aspdotnet: CreateApplicationHost "
                                   L"returned NULL, perhaps the AspNetMount "
                                   L"paths were invalid", APLOG_ERR, 0, s);
                }
            }
            catch (Exception *e) {
                    server_rec __nogc *s = (server_rec __nogc *)
                        (*__try_cast<__box int*>(hostSvr->get_Item(HostKey)));
                LogServerError(L"mod_aspdotnet: CreateApplicationHost failed, "
                               L"Exception follows", APLOG_ERR, 0, s);
                LogServerError(e->ToString(), APLOG_ERR, 0, s);
                hostObj->set_Item(HostKey, NULL);
            }
            return false;
        }

        void HostFinalized(int HostKey, Host *thisHost)
        {
            // Destroy only the correct host
            Threading::Monitor::Enter(this);
            if (thisHost->Equals(hostObj->get_Item(HostKey)))
            {
                try {
                    hostObj->set_Item(HostKey, NULL);
                }
                catch (Exception * /*e*/) {
                }
            }
            Threading::Monitor::Exit(this);
        }

        // Instantiate a new Host object; trailing slashes are acceptable
        int CreateHost(String* virtualPath, String* physicalPath, int Server) 
        {
            int newHostKey;

            physicalPath = physicalPath->Replace(L'/', L'\\');

            newHostKey = hostObj->get_Count();
            hostSvr->Add(__box(Server));
            hostURI->Add(virtualPath);
            hostDir->Add(physicalPath);
            hostObj->Add(NULL);

            if (!ConnectHost(newHostKey)) {
                server_rec __nogc *s = (server_rec __nogc *)Server;
                String *msg = String::Concat(L"Mount failure creating host ",
                          __try_cast<String*>(hostURI->get_Item(newHostKey)),
                                             L" mapped to ",
                          __try_cast<String*>(hostDir->get_Item(newHostKey)));                                                
                LogServerError(msg, APLOG_ERR, 0, s);
                newHostKey = -1;
            }

            return newHostKey;
        }

        int HandleHostRequest(int HostKey, int Req) {
            request_rec __nogc *rr = (request_rec __nogc *)Req;
            String* msg;

            Object* obj = hostObj->get_Item(HostKey);
            if (obj != NULL) {
                try {
                    Host* host = __try_cast<Host*>(obj);
                    int status = host->HandleRequest(Req);
                    return status;
                }
                // Presume all Remoting/AppDomainUnloadedExceptions 
                // require us to recreate the Host.  Fall through...
                catch (System::Runtime::Remoting::RemotingException *e) {
                    msg = String::Concat(L"RemotingException; ",
                                         e->ToString());
                    LogServerError(msg, APLOG_DEBUG, 0, rr->server);
                }
                catch (System::AppDomainUnloadedException *e) {
                    msg = String::Concat(L"AppDomainUnloadedException; ",
                                         e->ToString());
                    LogServerError(msg, APLOG_DEBUG, 0, rr->server);
                }
                catch (Exception *e) {
                    msg = String::Concat(L"HandleRequest: Failed with "
                                         L"unexpected Exception; ",
                                         e->ToString());                                                
                    LogServerError(msg, APLOG_ERR, 0, rr->server);
                    return 500;
                }
            }

            msg = String::Concat(L"Restarting host of ",
                       __try_cast<String*>(hostURI->get_Item(HostKey)),
                                 L" mapped to ",
                       __try_cast<String*>(hostDir->get_Item(HostKey)));
            LogServerError(msg, APLOG_INFO, 0, rr->server);

            int welocked = 0;
            try {
                Threading::Monitor::Enter(this);
                welocked = 1;
                Object* tryobj = hostObj->get_Item(HostKey);
                if (tryobj == obj)
                {
                    // Release all stale obj references and catch any
                    // sort of dispatch exceptions (and ignore them).
                    try {
                        hostObj->set_Item(HostKey, NULL);
                    }
                    catch (Exception * /*e*/) {
                    }
                    try {
                        obj = tryobj;
                    }
                    catch (Exception * /*e*/) {
                    }
                }
                if (obj == NULL) {
                    if (!ConnectHost(HostKey) || !(obj = hostObj->get_Item(HostKey))) {
                        Threading::Monitor::Exit(this);
                        msg = String::Concat(L"Mount failure restarting host ",
                                   __try_cast<String*>(hostURI->get_Item(HostKey)),
                                             L" mapped to ",
                                   __try_cast<String*>(hostDir->get_Item(HostKey)));                                                
                        LogServerError(msg, APLOG_ERR, 0, rr->server);
                        return 500;
                    }
                }
                Threading::Monitor::Exit(this);
            }
            catch (Exception *e) {
                if (welocked) {
                    Threading::Monitor::Exit(this);
                }
                msg = String::Concat(L"Mount failure restarting host ",
                           __try_cast<String*>(hostURI->get_Item(HostKey)),
                                     L" mapped to ",
                           __try_cast<String*>(hostDir->get_Item(HostKey)));
                LogServerError(msg, APLOG_ERR, 0, rr->server);
                msg = String::Concat(L"Mount failure exception restarting "
                                     L"host; ", e->ToString());
                LogServerError(msg, APLOG_ERR, 0, rr->server);
                return 500;
            }

            try {
                Host* host = __try_cast<Host*>(obj);
                int status = host->HandleRequest(Req);
                return status;
            }
            catch (Exception *e) {
                msg = String::Concat(L"RequestHandler; failure following "
                                     L"host restart; ", e->ToString());                                                
                LogServerError(msg, APLOG_ERR, 0, rr->server);
            }
            return 500;
        }

    protected:
        void ~HostFactory()
        {
            Destroy();
        }
    };
  }
}
